// Copyright ® 2025 OneSpan North America, Inc. All rights reserved. 

 
/////////////////////////////////////////////////////////////////////////////
//
//
// This file is example source code. It is provided for your information and
// assistance. See your licence agreement for details and the terms and
// conditions of the licence which governs the use of the source code. By using
// such source code you will be accepting these terms and conditions. If you do
// not wish to accept these terms and conditions, DO NOT OPEN THE FILE OR USE
// THE SOURCE CODE.
//
// Note that there is NO WARRANTY.
//
//////////////////////////////////////////////////////////////////////////////


import UIKit
import AVFoundation
import MSSImageScanner

protocol UIKitSampleControllerDelegate: AnyObject {
    func didReceiveResult(_ scanResult: ScanResult)
}

class UIKitSampleController: UIViewController {
    private let cameraManager = CameraManager()
    private let context = CIContext()
    
    private var previewLayer: AVCaptureVideoPreviewLayer?
    private var overlayView: OverlayView!
    private var headerBarView: HeaderBarView!

    weak var delegate: UIKitSampleControllerDelegate?
    
    override func viewDidLoad() {
        super.viewDidLoad()
        
        cameraManager.delegate = self
        setupPreviewLayer()
        setupOverlay()
    }
    
    override func viewDidLayoutSubviews() {
        super.viewDidLayoutSubviews()
        previewLayer?.frame = UIScreen.main.bounds
        overlayView.updateFrameAndRedraw(UIScreen.main.bounds)
    }
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        cameraManager.startSession()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        super.viewDidDisappear(animated)
        cameraManager.stopSession()
    }
    
    private func setupPreviewLayer() {
        let previewLayer = AVCaptureVideoPreviewLayer(session: cameraManager.session)
        previewLayer.videoGravity = .resizeAspectFill
        view.layer.addSublayer(previewLayer)
        self.previewLayer = previewLayer
    }
    
    private func setupOverlay() {
        overlayView = OverlayView(frame: UIScreen.main.bounds,
                                  overlayColor: UIColor.black.withAlphaComponent(0.6))
        overlayView.translatesAutoresizingMaskIntoConstraints = false
        view.addSubview(overlayView)
        
        NSLayoutConstraint.activate([
            overlayView.topAnchor.constraint(equalTo: view.topAnchor),
            overlayView.bottomAnchor.constraint(equalTo: view.bottomAnchor),
            overlayView.leadingAnchor.constraint(equalTo: view.leadingAnchor),
            overlayView.trailingAnchor.constraint(equalTo: view.trailingAnchor)
        ])
        
        headerBarView = HeaderBarView()
        headerBarView.translatesAutoresizingMaskIntoConstraints = false
        headerBarView.onCloseTapped = { [weak self] in
            self?.delegate?.didReceiveResult(.cancelled)
        }
        view.addSubview(headerBarView)
        
        NSLayoutConstraint.activate([
            headerBarView.topAnchor.constraint(equalTo: view.topAnchor),
            headerBarView.leadingAnchor.constraint(equalTo: view.leadingAnchor),
            headerBarView.trailingAnchor.constraint(equalTo: view.trailingAnchor),
            headerBarView.heightAnchor.constraint(equalToConstant: 110)
        ])
    }
}

extension UIKitSampleController: CameraManagerDelegate {
    func cameraManager(didOutput sampleBuffer: CMSampleBuffer) {
        guard let uiImage = makeUIImage(from: sampleBuffer) else {
            return
        }
        
        DispatchQueue.global().async {
            self.decodeImage(uiImage)
        }
    }
    
    func cameraManager(didFailWithError error: CameraManagerError) {
        DispatchQueue.main.async {
            self.headerBarView.configure(with: error.errorDescription)
        }
    }
}

extension UIKitSampleController {
    /// Used to convert `CMSampleBuffer` data from `AVFoundation` to `UIImage` object
    private func makeUIImage(from sampleBuffer: CMSampleBuffer?) -> UIImage? {
        guard
            let sampleBuffer,
            let cvBuffer = CMSampleBufferGetImageBuffer(sampleBuffer) else {
            return nil
        }
        
        let ciImage = CIImage(cvPixelBuffer: cvBuffer)
        
        guard let cgImage = context.createCGImage(ciImage, from: ciImage.extent) else {
            return nil
        }
        
        return UIImage(cgImage: cgImage)
    }
    
    /// Actual `UIImage` decoding stage
    private func decodeImage(_ image: UIImage) {
        do {
            let output = try QRCodeScannerSDK.decodeImage(image, codeType: .all)
            DispatchQueue.main.async {
                self.delegate?.didReceiveResult(.scanned(output.result, codeType: output.codeType))
            }
        } catch ScannerError.invalidImage {
            // Code not decoded - try in next frame
        } catch let error {
            DispatchQueue.main.async {
                self.delegate?.didReceiveResult(.error(error))
            }
        }
    }
}
